<?php

class Database {
    private static $instance = null;
    private $connection;
    private $lastQuery;
    private $lastError;

    private function __construct() {
        try {
            $this->connection = new mysqli(
                DB_HOST,
                DB_USER,
                DB_PASS,
                DB_NAME,
                DB_PORT
            );

            if ($this->connection->connect_error) {
                throw new Exception('Database Connection Error: ' . $this->connection->connect_error);
            }

            $this->connection->set_charset(DB_CHARSET);
        } catch (Exception $e) {
            die('Error: ' . $e->getMessage());
        }
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function getConnection() {
        return $this->connection;
    }

    public function query($sql) {
        $this->lastQuery = $sql;
        $result = $this->connection->query($sql);
        
        if (!$result) {
            $this->lastError = $this->connection->error;
            return false;
        }
        
        return $result;
    }

    public function prepare($sql) {
        $stmt = $this->connection->prepare($sql);
        if (!$stmt) {
            $this->lastError = $this->connection->error;
            return false;
        }
        return $stmt;
    }

    public function fetch($sql, $types = '', $params = []) {
        $stmt = $this->prepare($sql);
        if (!$stmt) return null;

        if (!empty($params)) {
            call_user_func_array([$stmt, 'bind_param'], array_merge([$types], $params));
        }

        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();

        return $data;
    }

    public function fetchAll($sql, $types = '', $params = []) {
        $stmt = $this->prepare($sql);
        if (!$stmt) return [];

        if (!empty($params)) {
            call_user_func_array([$stmt, 'bind_param'], array_merge([$types], $params));
        }

        $stmt->execute();
        $result = $stmt->get_result();
        $data = [];
        
        while ($row = $result->fetch_assoc()) {
            $data[] = $row;
        }
        
        $stmt->close();
        return $data;
    }

    public function insert($table, $data) {
        $columns = array_keys($data);
        $values = array_values($data);
        $placeholders = array_fill(0, count($values), '?');
        
        $types = '';
        foreach ($values as $value) {
            $types .= $this->getType($value);
        }
        
        $sql = "INSERT INTO {$table} (" . implode(',', $columns) . ") VALUES (" . implode(',', $placeholders) . ")";
        $stmt = $this->prepare($sql);
        
        if (!$stmt) return false;
        
        call_user_func_array([$stmt, 'bind_param'], array_merge([$types], $values));
        $result = $stmt->execute();
        $stmt->close();
        
        return $result;
    }

    public function update($table, $data, $where, $whereTypes = '', $whereParams = []) {
        $updates = [];
        $types = '';
        $params = [];
        
        foreach ($data as $column => $value) {
            $updates[] = "{$column} = ?";
            $types .= $this->getType($value);
            $params[] = $value;
        }
        
        $types .= $whereTypes;
        $params = array_merge($params, $whereParams);
        
        $sql = "UPDATE {$table} SET " . implode(', ', $updates) . " WHERE {$where}";
        $stmt = $this->prepare($sql);
        
        if (!$stmt) return false;
        
        call_user_func_array([$stmt, 'bind_param'], array_merge([$types], $params));
        $result = $stmt->execute();
        $stmt->close();
        
        return $result;
    }

    public function delete($table, $where, $types = '', $params = []) {
        $sql = "DELETE FROM {$table} WHERE {$where}";
        $stmt = $this->prepare($sql);
        
        if (!$stmt) return false;
        
        if (!empty($params)) {
            call_user_func_array([$stmt, 'bind_param'], array_merge([$types], $params));
        }
        
        $result = $stmt->execute();
        $stmt->close();
        
        return $result;
    }

    public function count($table, $where = '', $types = '', $params = []) {
        $sql = "SELECT COUNT(*) as total FROM {$table}";
        
        if (!empty($where)) {
            $sql .= " WHERE {$where}";
        }
        
        if ($where) {
            $result = $this->fetch($sql, $types, $params);
        } else {
            $result = $this->fetch($sql);
        }
        
        return $result ? $result['total'] : 0;
    }

    private function getType($value) {
        if (is_int($value)) {
            return 'i';
        } elseif (is_float($value)) {
            return 'd';
        } else {
            return 's';
        }
    }

    public function getLastError() {
        return $this->lastError;
    }

    public function getLastQuery() {
        return $this->lastQuery;
    }

    public function beginTransaction() {
        return $this->connection->begin_transaction();
    }

    public function commit() {
        return $this->connection->commit();
    }

    public function rollback() {
        return $this->connection->rollback();
    }

    public function close() {
        if ($this->connection) {
            $this->connection->close();
        }
    }

    public function __destruct() {
        $this->close();
    }
}
