<?php

class Cache {
    private $db;
    private $userId;

    public function __construct($userId = null) {
        $this->db = Database::getInstance();
        $this->userId = $userId;
    }

    /**
     * Get value from cache
     */
    public function get($key) {
        if (!$this->userId) return null;

        $data = $this->db->fetch(
            "SELECT cache_value FROM api_cache 
             WHERE admin_id = ? AND cache_key = ? AND expires_at > NOW()",
            "is",
            [&$this->userId, &$key]
        );

        if (!$data) {
            return null;
        }

        return json_decode($data['cache_value'], true);
    }

    /**
     * Set value to cache
     */
    public function set($key, $value, $duration = 3600) {
        if (!$this->userId) return false;

        $expiresAt = date('Y-m-d H:i:s', time() + $duration);
        $cacheValue = json_encode($value);

        // Try update if exists
        $existing = $this->db->fetch(
            "SELECT id FROM api_cache WHERE admin_id = ? AND cache_key = ?",
            "is",
            [&$this->userId, &$key]
        );

        if ($existing) {
            return $this->db->update(
                'api_cache',
                ['cache_value' => $cacheValue, 'expires_at' => $expiresAt],
                'admin_id = ? AND cache_key = ?',
                'iss',
                [&$this->userId, &$key, &$expiresAt]
            );
        } else {
            return $this->db->insert('api_cache', [
                'admin_id' => $this->userId,
                'cache_key' => $key,
                'cache_value' => $cacheValue,
                'expires_at' => $expiresAt
            ]);
        }
    }

    /**
     * Delete cache by key
     */
    public function delete($key) {
        if (!$this->userId) return false;

        return $this->db->delete(
            'api_cache',
            'admin_id = ? AND cache_key = ?',
            'is',
            [&$this->userId, &$key]
        );
    }

    /**
     * Delete cache by pattern (e.g., 'produk_list_*')
     */
    public function deletePattern($pattern) {
        if (!$this->userId) return false;

        $pattern = str_replace('*', '%', $pattern);
        return $this->db->delete(
            'api_cache',
            'admin_id = ? AND cache_key LIKE ?',
            'is',
            [&$this->userId, &$pattern]
        );
    }

    /**
     * Clear all cache for user
     */
    public function clearAll() {
        if (!$this->userId) return false;

        return $this->db->delete(
            'api_cache',
            'admin_id = ?',
            'i',
            [&$this->userId]
        );
    }

    /**
     * Clear expired cache (cleanup)
     */
    public function clearExpired() {
        return $this->db->query("DELETE FROM api_cache WHERE expires_at <= NOW()");
    }

    /**
     * Check if cache exists and not expired
     */
    public function exists($key) {
        if (!$this->userId) return false;

        $result = $this->db->fetch(
            "SELECT id FROM api_cache 
             WHERE admin_id = ? AND cache_key = ? AND expires_at > NOW()",
            "is",
            [&$this->userId, &$key]
        );

        return !empty($result);
    }
}
