<?php

class CSVHandler {
    private $filePath;
    private $headers = [];
    private $data = [];
    private $errors = [];

    public function __construct($filePath) {
        $this->filePath = $filePath;
    }

    /**
     * Parse CSV file
     */
    public function parse() {
        if (!file_exists($this->filePath)) {
            $this->errors[] = 'File tidak ditemukan';
            return false;
        }

        if (!in_array(pathinfo($this->filePath, PATHINFO_EXTENSION), ['csv', 'txt'])) {
            $this->errors[] = 'File harus berformat CSV';
            return false;
        }

        $file = fopen($this->filePath, 'r');
        if (!$file) {
            $this->errors[] = 'Gagal membuka file';
            return false;
        }

        $row = 0;
        while (($data = fgetcsv($file, 0, ',', '"')) !== false) {
            if ($row === 0) {
                // Header row
                $this->headers = array_map('trim', $data);
            } else {
                // Data rows
                $record = [];
                foreach ($this->headers as $index => $header) {
                    $record[$header] = trim($data[$index] ?? '');
                }
                $this->data[] = $record;
            }
            $row++;
        }

        fclose($file);

        return count($this->data) > 0;
    }

    /**
     * Parse Excel file (XLSX)
     */
    public function parseExcel() {
        if (!file_exists($this->filePath)) {
            $this->errors[] = 'File tidak ditemukan';
            return false;
        }

        $ext = pathinfo($this->filePath, PATHINFO_EXTENSION);
        if (!in_array($ext, ['xlsx', 'xls'])) {
            $this->errors[] = 'File harus berformat XLSX atau XLS';
            return false;
        }

        try {
            // Untuk production, gunakan library seperti PHPExcel atau OpenSpout
            // Untuk sekarang kita konversi ke CSV dulu
            if ($ext === 'xlsx') {
                // Gunakan command line atau library
                $csvFile = str_replace('.xlsx', '.csv', $this->filePath);
                $this->convertXlsxToCsv($this->filePath, $csvFile);
                $this->filePath = $csvFile;
                return $this->parse();
            }
        } catch (Exception $e) {
            $this->errors[] = 'Gagal parsing file: ' . $e->getMessage();
            return false;
        }

        return true;
    }

    /**
     * Validate data
     */
    public function validate() {
        $requiredFields = [
            'id_kategori',
            'id_lokasi',
            'jenis_barang',
            'nama_barang',
            'deskripsi_singkat',
            'deskripsi_panjang',
            'harga_barang',
            'stok_barang',
            'url_gambar_barang_1'
        ];

        // Check headers
        foreach ($requiredFields as $field) {
            if (!in_array($field, $this->headers)) {
                $this->errors[] = "Kolom '{$field}' tidak ditemukan di file";
                return false;
            }
        }

        // Validate data
        foreach ($this->data as $index => $row) {
            $rowNum = $index + 2; // +2 because of header

            // Check required fields
            foreach ($requiredFields as $field) {
                if (empty($row[$field])) {
                    $this->errors[] = "Baris {$rowNum}: {$field} tidak boleh kosong";
                    return false;
                }
            }

            // Validate numeric fields
            if (!is_numeric($row['harga_barang'])) {
                $this->errors[] = "Baris {$rowNum}: harga_barang harus berupa angka";
                return false;
            }

            if (!is_numeric($row['stok_barang'])) {
                $this->errors[] = "Baris {$rowNum}: stok_barang harus berupa angka";
                return false;
            }

            // Validate jenis_barang
            if (!in_array($row['jenis_barang'], ['fisik', 'digital'])) {
                $this->errors[] = "Baris {$rowNum}: jenis_barang harus 'fisik' atau 'digital'";
                return false;
            }

            // Validate URL
            if (!filter_var($row['url_gambar_barang_1'], FILTER_VALIDATE_URL)) {
                $this->errors[] = "Baris {$rowNum}: url_gambar_barang_1 bukan URL yang valid";
                return false;
            }
        }

        return true;
    }

    /**
     * Get parsed data
     */
    public function getData() {
        return $this->data;
    }

    /**
     * Get headers
     */
    public function getHeaders() {
        return $this->headers;
    }

    /**
     * Get errors
     */
    public function getErrors() {
        return $this->errors;
    }

    /**
     * Get row count
     */
    public function getRowCount() {
        return count($this->data);
    }

    /**
     * Convert XLSX to CSV
     */
    private function convertXlsxToCsv($xlsxFile, $csvFile) {
        // This is a simple approach - for production use PhpSpreadsheet
        $zip = new ZipArchive();
        
        if (!$zip->open($xlsxFile)) {
            throw new Exception('Gagal membuka file XLSX');
        }

        $xmlData = $zip->getFromName('xl/worksheets/sheet1.xml');
        $zip->close();

        if ($xmlData === false) {
            throw new Exception('Sheet tidak ditemukan di file XLSX');
        }

        // Parse XML dan convert ke CSV
        $xml = simplexml_load_string($xmlData);
        
        $csv = [];
        foreach ($xml->sheetData->row as $row) {
            $rowData = [];
            foreach ($row->c as $cell) {
                $value = (string)$cell->v;
                
                // Handle shared strings
                $cellType = (string)$cell->attributes()->t;
                if ($cellType === 's') {
                    $stringIndex = (int)$value;
                    $sharedStrings = simplexml_load_string($zip->getFromName('xl/sharedStrings.xml'));
                    $value = (string)$sharedStrings->si[$stringIndex]->t;
                }
                
                $rowData[] = $value;
            }
            $csv[] = $rowData;
        }

        // Write CSV
        $fp = fopen($csvFile, 'w');
        foreach ($csv as $row) {
            fputcsv($fp, $row);
        }
        fclose($fp);
    }

    /**
     * Generate template CSV
     */
    public static function generateTemplate() {
        $headers = [
            'id_kategori',
            'id_lokasi',
            'jenis_barang',
            'nama_barang',
            'deskripsi_singkat',
            'deskripsi_panjang',
            'harga_barang',
            'stok_barang',
            'url_gambar_barang_1',
            'url_gambar_barang_2',
            'harga_modal',
            'harga_barang_asli',
            'sku'
        ];

        $sample = [
            ['26324', '18696', 'digital', 'Produk Digital 1', 'Deskripsi singkat', 'Deskripsi panjang produk', '23000', '100', 'https://example.com/image1.jpg', '', '20000', '25000', 'SKU001'],
            ['26324', '18696', 'fisik', 'Produk Fisik 1', 'Deskripsi singkat', 'Deskripsi panjang produk', '50000', '20', 'https://example.com/image2.jpg', '', '40000', '55000', 'SKU002']
        ];

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=template_produk.csv');

        $output = fopen('php://output', 'w');
        
        // Write BOM untuk support Excel
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        fputcsv($output, $headers);
        foreach ($sample as $row) {
            fputcsv($output, $row);
        }

        fclose($output);
        exit;
    }
}
