<?php

class BukaOlshopAPI {
    private $apiKey;
    private $baseUrl = BUKAOLSHOP_API_BASE;
    private $cache;
    private $userId;

    public function __construct($apiKey, $userId = null) {
        $this->apiKey = $apiKey;
        $this->userId = $userId;
        $this->cache = new Cache($userId);
    }

    /**
     * Get ID Kategori dan Lokasi
     */
    public function getKategoriDanLokasi($forceRefresh = false) {
        $cacheKey = 'kategori_lokasi';

        if (!$forceRefresh) {
            $cached = $this->cache->get($cacheKey);
            if ($cached) {
                return $cached;
            }
        }

        $response = $this->makeRequest('GET', '/produk/id_upload');

        if ($response['success']) {
            // Simpan ke cache
            $this->cache->set($cacheKey, $response['data'], API_CACHE_DURATION);

            // Simpan ke database
            if ($this->userId) {
                $this->saveKategoriDanLokasiToDB($response['data']);
            }

            return $response['data'];
        }

        // Jika API gagal, coba ambil dari DB
        if ($this->userId) {
            $dbData = $this->getKategoriDanLokasiFromDB();
            if (!empty($dbData['kategori']) || !empty($dbData['lokasi'])) {
                return $dbData;
            }
        }

        return $response;
    }

    /**
     * Upload Produk
     */
    public function uploadProduk($data) {
        // Validasi required fields
        $required = ['id_kategori', 'id_lokasi', 'jenis_barang', 'nama_barang', 
                     'deskripsi_singkat', 'deskripsi_panjang', 'harga_barang', 'stok_barang', 'url_gambar_barang_1'];
        
        foreach ($required as $field) {
            if (empty($data[$field])) {
                return ['success' => false, 'message' => "Field {$field} wajib diisi"];
            }
        }

        // Validasi harga dan stok harus number
        if (!is_numeric($data['harga_barang']) || !is_numeric($data['stok_barang'])) {
            return ['success' => false, 'message' => 'Harga dan stok harus berupa angka'];
        }

        $response = $this->makeRequest('POST', '/produk/create', $data);

        if ($response['success']) {
            // Simpan ke upload queue
            if ($this->userId) {
                $this->saveToUploadQueue($data, $response['data']);
            }
        }

        return $response;
    }

    /**
     * Get Daftar Produk
     */
    public function getDaftarProduk($page = 1, $forceRefresh = false) {
        if ($page < 1) {
            return ['success' => false, 'message' => 'Page harus lebih dari 0'];
        }

        $cacheKey = 'produk_list_' . $page;

        if (!$forceRefresh) {
            $cached = $this->cache->get($cacheKey);
            if ($cached) {
                return $cached;
            }
        }

        $response = $this->makeRequest('GET', '/produk/list', ['page' => $page]);

        if ($response['success']) {
            $this->cache->set($cacheKey, $response['data'], API_CACHE_DURATION);
        }

        return $response;
    }

    /**
     * Get Detail Produk
     */
    public function getDetailProduk($id) {
        $cacheKey = 'produk_' . $id;

        $cached = $this->cache->get($cacheKey);
        if ($cached) {
            return $cached;
        }

        $response = $this->makeRequest('GET', '/produk/' . $id);

        if ($response['success']) {
            $this->cache->set($cacheKey, $response['data'], API_CACHE_DURATION);
        }

        return $response;
    }

    /**
     * Edit Produk
     */
    public function editProduk($data) {
        if (empty($data['id_barang'])) {
            return ['success' => false, 'message' => 'ID barang wajib diisi'];
        }

        // Hanya boleh update fields yang allowed
        $allowed = ['nama_barang', 'deskripsi_panjang', 'deskripsi_singkat', 'berat', 
                    'jumlah_stok', 'kondisi', 'merek', 'dalam_kotak', 'garansi', 
                    'harga_barang', 'harga_barang_asli', 'sku'];
        
        $updateData = ['id_barang' => $data['id_barang']];
        
        foreach ($allowed as $field) {
            if (isset($data[$field])) {
                $updateData[$field] = $data[$field];
            }
        }

        // Minimal harus ada 1 field yang diupdate
        if (count($updateData) === 1) {
            return ['success' => false, 'message' => 'Minimal ada 1 field yang harus diupdate'];
        }

        $response = $this->makeRequest('PATCH', '/produk/id', $updateData);

        if ($response['success']) {
            // Clear cache produk ini
            $this->cache->delete('produk_' . $data['id_barang']);
            // Clear list cache
            $this->cache->deletePattern('produk_list_');
        }

        return $response;
    }

    /**
     * Make HTTP Request ke API BukaOlshop
     */
    private function makeRequest($method, $endpoint, $data = null) {
        $url = $this->baseUrl . $endpoint;
        
        // Tambah query parameter untuk GET
        if ($method === 'GET' && is_array($data)) {
            $url .= '?' . http_build_query($data);
        }

        $headers = [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
            'User-Agent: BukaOlshopAdmin/1.0'
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, API_TIMEOUT);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        } elseif ($method === 'PATCH') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return ['success' => false, 'message' => 'Connection error: ' . $error];
        }

        $decoded = json_decode($response, true);

        if ($httpCode === 401) {
            return ['success' => false, 'message' => 'API Key tidak valid (401 Unauthorized)'];
        }

        if ($httpCode >= 400) {
            return [
                'success' => false,
                'message' => $decoded['status'] ?? "Request gagal (HTTP {$httpCode})",
                'code' => $httpCode
            ];
        }

        if ($decoded && isset($decoded['code']) && $decoded['code'] === 200) {
            return ['success' => true, 'data' => $decoded];
        } else {
            return ['success' => false, 'message' => $decoded['status'] ?? 'Unknown error'];
        }
    }

    /**
     * Save Kategori dan Lokasi to Database
     */
    private function saveKategoriDanLokasiToDB($data) {
        if (!$this->userId) return;

        $db = Database::getInstance();

        // Hapus kategori lama
        $db->delete('kategori', 'admin_id = ?', 'i', [&$this->userId]);

        // Insert kategori baru
        if (isset($data['kategori'])) {
            foreach ($data['kategori'] as $kat) {
                $db->insert('kategori', [
                    'id_kategori' => $kat['id_kategori'],
                    'nama_kategori' => $kat['nama_kategori'],
                    'admin_id' => $this->userId
                ]);
            }
        }

        // Hapus lokasi lama
        $db->delete('lokasi', 'admin_id = ?', 'i', [&$this->userId]);

        // Insert lokasi baru
        if (isset($data['lokasi'])) {
            foreach ($data['lokasi'] as $lok) {
                $db->insert('lokasi', [
                    'id_lokasi' => $lok['id_lokasi'],
                    'nama_pengirim' => $lok['nama_pengirim'],
                    'tipe_lokasi' => $lok['tipe_lokasi'] ?? '',
                    'admin_id' => $this->userId
                ]);
            }
        }
    }

    /**
     * Get Kategori dan Lokasi dari Database
     */
    private function getKategoriDanLokasiFromDB() {
        if (!$this->userId) return ['kategori' => [], 'lokasi' => []];

        $db = Database::getInstance();

        $kategori = $db->fetchAll(
            "SELECT id_kategori, nama_kategori FROM kategori WHERE admin_id = ? ORDER BY nama_kategori",
            "i",
            [&$this->userId]
        );

        $lokasi = $db->fetchAll(
            "SELECT id_lokasi, nama_pengirim, tipe_lokasi FROM lokasi WHERE admin_id = ? ORDER BY nama_pengirim",
            "i",
            [&$this->userId]
        );

        return ['kategori' => $kategori, 'lokasi' => $lokasi];
    }

    /**
     * Save ke Upload Queue
     */
    private function saveToUploadQueue($data, $response) {
        if (!$this->userId) return;

        $db = Database::getInstance();

        $db->insert('upload_queue', [
            'id_antrian' => $response['id_antrian'] ?? '',
            'id_kategori' => $data['id_kategori'],
            'id_lokasi' => $data['id_lokasi'],
            'nama_barang' => $data['nama_barang'],
            'harga_barang' => $data['harga_barang'],
            'stok_barang' => $data['stok_barang'],
            'jenis_barang' => $data['jenis_barang'],
            'url_gambar_barang_1' => $data['url_gambar_barang_1'],
            'status' => 'pending',
            'response_data' => json_encode($response),
            'admin_id' => $this->userId
        ]);
    }

    /**
     * Clear Cache for Upload
     */
    public function clearUploadCache() {
        $this->cache->deletePattern('produk_list_');
        $this->cache->deletePattern('produk_');
    }
}
