<?php

class Auth {
    private $db;
    private $sessionTimeout = SESSION_TIMEOUT;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function register($username, $email, $password, $api_key) {
        // Validasi input
        if (empty($username) || empty($email) || empty($password) || empty($api_key)) {
            return ['success' => false, 'message' => 'Semua field wajib diisi'];
        }

        if (strlen($password) < 6) {
            return ['success' => false, 'message' => 'Password minimal 6 karakter'];
        }

        // Check username exist
        $existing = $this->db->fetch(
            "SELECT id FROM admin_users WHERE username = ?",
            "s",
            [&$username]
        );

        if ($existing) {
            return ['success' => false, 'message' => 'Username sudah terdaftar'];
        }

        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_BCRYPT, ['cost' => 10]);

        // Insert user
        $result = $this->db->insert('admin_users', [
            'username' => $username,
            'email' => $email,
            'password' => $hashedPassword,
            'api_key' => $api_key,
            'is_active' => 1
        ]);

        if ($result) {
            return ['success' => true, 'message' => 'Registrasi berhasil'];
        } else {
            return ['success' => false, 'message' => 'Gagal registrasi: ' . $this->db->getLastError()];
        }
    }

    public function login($username, $password) {
        if (empty($username) || empty($password)) {
            return ['success' => false, 'message' => 'Username dan password wajib diisi'];
        }

        $user = $this->db->fetch(
            "SELECT id, username, password, api_key, is_active FROM admin_users WHERE username = ?",
            "s",
            [&$username]
        );

        if (!$user) {
            return ['success' => false, 'message' => 'Username atau password salah'];
        }

        if (!$user['is_active']) {
            return ['success' => false, 'message' => 'Akun Anda tidak aktif'];
        }

        if (!password_verify($password, $user['password'])) {
            return ['success' => false, 'message' => 'Username atau password salah'];
        }

        // Set session
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['api_key'] = $user['api_key'];
        $_SESSION['login_time'] = time();
        $_SESSION['ip_address'] = $this->getClientIP();

        // Log activity
        $this->logActivity($user['id'], 'LOGIN', 'Admin login', $this->getClientIP());

        return ['success' => true, 'message' => 'Login berhasil'];
    }

    public function logout() {
        if (isset($_SESSION['user_id'])) {
            $this->logActivity($_SESSION['user_id'], 'LOGOUT', 'Admin logout', $this->getClientIP());
        }

        session_destroy();
        return ['success' => true, 'message' => 'Logout berhasil'];
    }

    public function isLoggedIn() {
        if (!isset($_SESSION['user_id'])) {
            return false;
        }

        // Check session timeout
        if (time() - $_SESSION['login_time'] > $this->sessionTimeout) {
            $this->logout();
            return false;
        }

        // Check IP address (security)
        if ($_SESSION['ip_address'] !== $this->getClientIP()) {
            $this->logout();
            return false;
        }

        // Refresh session timeout
        $_SESSION['login_time'] = time();

        return true;
    }

    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }

        return [
            'id' => $_SESSION['user_id'],
            'username' => $_SESSION['username'],
            'api_key' => $_SESSION['api_key']
        ];
    }

    public function getUserId() {
        return $_SESSION['user_id'] ?? null;
    }

    public function getApiKey() {
        return $_SESSION['api_key'] ?? null;
    }

    public function generateCSRFToken() {
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
            $_SESSION[CSRF_TOKEN_NAME . '_time'] = time();
        }

        return $_SESSION[CSRF_TOKEN_NAME];
    }

    public function verifyCSRFToken($token) {
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            return false;
        }

        if (time() - $_SESSION[CSRF_TOKEN_NAME . '_time'] > CSRF_TOKEN_EXPIRY) {
            unset($_SESSION[CSRF_TOKEN_NAME]);
            return false;
        }

        return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }

    private function logActivity($userId, $action, $description, $ipAddress) {
        $this->db->insert('activity_log', [
            'admin_id' => $userId,
            'action' => $action,
            'description' => $description,
            'ip_address' => $ipAddress,
            'created_at' => date('Y-m-d H:i:s')
        ]);
    }

    private function getClientIP() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } else {
            return $_SERVER['REMOTE_ADDR'];
        }
    }

    public function updatePassword($userId, $oldPassword, $newPassword) {
        if (empty($oldPassword) || empty($newPassword)) {
            return ['success' => false, 'message' => 'Password lama dan baru wajib diisi'];
        }

        if (strlen($newPassword) < 6) {
            return ['success' => false, 'message' => 'Password minimal 6 karakter'];
        }

        $user = $this->db->fetch(
            "SELECT password FROM admin_users WHERE id = ?",
            "i",
            [&$userId]
        );

        if (!$user) {
            return ['success' => false, 'message' => 'User tidak ditemukan'];
        }

        if (!password_verify($oldPassword, $user['password'])) {
            return ['success' => false, 'message' => 'Password lama salah'];
        }

        $hashedPassword = password_hash($newPassword, PASSWORD_BCRYPT, ['cost' => 10]);
        $result = $this->db->update('admin_users', ['password' => $hashedPassword], 'id = ?', 'i', [&$userId]);

        if ($result) {
            $this->logActivity($userId, 'UPDATE_PASSWORD', 'Admin mengubah password', $this->getClientIP());
            return ['success' => true, 'message' => 'Password berhasil diubah'];
        } else {
            return ['success' => false, 'message' => 'Gagal mengubah password'];
        }
    }
}
